<?php

namespace App\Model;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Config;
use App\Traits\QueryCacheable;

class Setting extends Model
{
    use QueryCacheable;

    /**
     * Specify the amount of time to cache queries.
     * Do not specify or set it to null to disable caching.
     *
     * @var int|\DateTime
     */
    public $cacheFor = 3600;

    /**
     * The tags for the query cache. Can be useful
     * if flushing cache for specific tags only.
     *
     * @var null|array
     */
    public $cacheTags = ['settings'];

    /**
     * Invalidate the cache automatically
     * upon update in the database.
     *
     * @var bool
     */
    protected static $flushCacheOnUpdate = true;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'key', 'value',
    ];

    public static $MANAGEABLE_KEYS = [
        'app.name',
        'app.description',
        'app.keywords',
        'app.mobile.version.android',
        'app.mobile.version.ios',

        'app.maintenance_mode_status',
        'app.maintenance_mode_message',

        // 'mail.default',
        'mail.mailers.smtp.host',
        'mail.mailers.smtp.port',
        'mail.mailers.smtp.encryption',
        'mail.from.address',
        'mail.from.name',

        'admin.notification_emails',

        'social.facebook',
        'social.instagram',
        'social.linkedin',
        'social.youtube',
    ];

    public static $SENSITIVE_KEYS = [
        'mail.mailers.smtp.username',
        'mail.mailers.smtp.password',

        'recaptcha.api_site_key',
        'recaptcha.api_secret_key',
    ];

    public static function set($key, $value)
    {
        $setting = self::updateOrCreate(
            ['key' => $key],
            ['value' => $value]
        );

        Config::set($setting->key, $setting->value);

        return $setting;
    }

    public static function remove($key)
    {
        $setting = self::firstWhere('key', $key);

        if ($setting) {
            $setting->delete();
        }

        return $setting;
    }

    public static function getAdminNotifyEmails()
    {
        $emails = preg_split("/[\r\n,]+/", config('admin.notification_emails'), -1, PREG_SPLIT_NO_EMPTY);
        $emails = array_map('trim', $emails);
        $new_emails = [];

        foreach ($emails as $email) {
            if (filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $new_emails[] = $email;
            }
        }

        return $new_emails;
    }
}
