<?php

namespace App\Model;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Modules\User\Entities\User;
use Yajra\DataTables\Facades\DataTables;

class AuditLogin extends Model
{
    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'admin_id',
        'user_id',
        'email',
        'ip_address',
        'has_remember_me',
        'has_failed',
        'throttle_hits',
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        //
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'has_remember_me' => 'boolean',
        'has_failed' => 'boolean',
        'updated_at' => 'datetime:Y-m-d H:i:s',
        'created_at' => 'datetime:Y-m-d H:i:s',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function scopeFilters($query, Request $request)
    {
        $query->when($request->has('result'), function ($query) use ($request) {
            return $query->where('has_failed', $request->result == 'failed');
        });

        $query->when($request->has('user'), function ($query) use ($request) {
            return $query->where('user_id', $request->user);
        });

        $query->when($request->has('range') && $request->filled('range'), function ($query) use ($request) {
            $date_range = explode(' to ', $request->range);

            if (count($date_range) == 2) {
                $from = new Carbon($date_range[0]);
                $to = new Carbon($date_range[1]);

                $query->whereBetween('created_at', [$from->startOfDay(), $to->endOfDay()]);
            } else if (count($date_range) == 1) {
                $query->whereDate('created_at', $date_range[0]);
            }

            return $query;
        });
    }

    public static function makeDataTable($source, $trash = false)
    {
        return DataTables::of($source)
            ->addColumn('user', function (self $row) {
                return view('admin.datatables.related', [
                    'row' => $row,
                    'relation' => 'user',
                    'route' => 'user.edit',
                    'label' => 'name'
                ]);
            })
            ->addColumn('remember', function (self $row) {
                return view('admin.datatables.status', ['active' => $row->has_remember_me, 'true' => 'Yes', 'false' => 'No']);
            })
            ->addColumn('result', function (self $row) {
                return view('admin.datatables.status', ['active' => !$row->has_failed, 'true' => 'Success', 'false' => 'Failed']);
            })
            ->addColumn('actions', function (self $row) use ($trash) {
                $actions = ['row' => $row];

                if ($trash) {
                    $actions['restore_route'] = 'audit-login.restore';
                } else {
                    $actions['update_route'] = 'audit-login.edit';
                    $actions['delete_route'] = 'audit-login.destroy';
                }

                return view('admin.datatables.actions', $actions);
            })
            // to fixed yajra weird timezone return
            ->editColumn('created_at', fn($row) => $row->created_at)
            ->editColumn('updated_at', fn($row) => $row->updated_at)
            ->editColumn('deleted_at', fn($row) => $row->deleted_at)
            ->make(true);
    }
}
