<?php

namespace App\Http\Controllers;

use App\Http\Requests\SettingRequest;
use App\Model\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Str;
use Symfony\Component\Mailer\Exception\TransportExceptionInterface;
use Symfony\Component\Mailer\Mailer;
use Symfony\Component\Mailer\Transport;
use Symfony\Component\Mime\Email;

class SettingController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth:admin', 'admin.active']);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $this->authorize('viewAny', Setting::class);

        return view('admin.setting.index', [
            //
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \App\Http\Requests\SettingRequest  $request
     * @return \Illuminate\Http\Response
     */
    public function update(SettingRequest $request)
    {
        $this->authorize('update', Setting::class);

        $inputs = [];

        foreach ($request->validated() as $key => $val) {
            $dotKey = (!Str::startsWith($key, '-')) ? str_replace('-', '.', $key) : $key;
            $inputs[$dotKey] = $val;
        }

        $request->replace($inputs);

        $maintenance = $request->get('app.maintenance_mode_status');

        foreach ($request->all() as $key => $input) {
            if (in_array($key, Setting::$SENSITIVE_KEYS) && !is_null($input)) {
                Setting::set($key, $input);
            }

            if (in_array($key, Setting::$MANAGEABLE_KEYS)) {
                Setting::set($key, $input);
            }

            if ($key == 'app.logo') {
                if ($input = upload_file($request, 'app.logo', 'logo')) {
                    // uno()->deleteOpenGraphImage();
                    uno()->setLogo($input);
                    Setting::set($key, $input);
                }
            }
        }

        if ($maintenance === 'enable' && !app()->isDownForMaintenance()) {
            $this->toggleMaintenanceMode('down');
        } else if ($maintenance === 'disable' && app()->isDownForMaintenance()) {
            $this->toggleMaintenanceMode('up');
        }

        return redirect()->route('setting.index')->withStatus(__('Settings successfully updated.'));
    }

    public function testSMTPSendEmail(Request $request)
    {
        $name = $request->get('name') ? $request->name : config('mail.from.name');
        $address = $request->get('email') ? $request->email : config('mail.from.address');
        $username = $request->get('username') ? $request->username : config('mail.mailers.smtp.username');
        $password = $request->get('password') ? $request->password : config('mail.mailers.smtp.password');
        $encoded_password = rawurlencode($password);

        try {
            // smtp://username:password@hostname:port
            $transport = Transport::fromDsn("smtp://{$username}:{$encoded_password}@{$request->host}:{$request->port}");
            $mailer = new Mailer($transport);

            $email = (new Email());

            $email->from($address);
            $email->to($address);
            $email->text('test smtp send email');

            $mailer->send($email);

            if ($request->save) {
                Setting::set('mail.mailers.smtp.host', $request->host);
                Setting::set('mail.mailers.smtp.port', $request->port);
                Setting::set('mail.mailers.smtp.encryption', $request->encryption);
                Setting::set('mail.mailers.smtp.username', $username);
                Setting::set('mail.mailers.smtp.password', $password);
                Setting::set('mail.from.name', $name);
                Setting::set('mail.from.address', $address);

                return response()->json([
                    'status' => 1,
                    'message' => 'SMTP settings working successfully! The configuration has been saved.'
                ]);
            }

            return response()->json([
                'status' => 1,
                'message' => 'SMTP settings working successfully!'
            ]);
        } catch (TransportExceptionInterface | \Exception $e) {
            $message = $e->getMessage();
        }

        return response()->json([
            'status' => 0,
            'message' => $message
        ]);
    }

    private function toggleMaintenanceMode($status)
    {
        Artisan::call($status);

        return Artisan::output();
    }
}
