<?php

namespace App\Http\Controllers;

use App\Model\Banner;
use App\Http\Requests\BannerRequest;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Validator;

class BannerController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth:admin', 'admin.active']);
        $this->authorizeResource(Banner::class, 'banner');
    }

    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = Banner::query()
                ->filters($request);

            return Banner::makeDataTable($data);
        }

        return view('admin.banner.index');
    }

    public function trashed(Request $request)
    {
        $this->authorize('restore', Banner::class);

        if ($request->ajax()) {
            $data = Banner::onlyTrashed();

            return Banner::makeDataTable($data, trash: true);
        }

        return view('admin.banner.trashed');
    }

    /**
     * Show the form for creating a new resource.
     * @return Renderable
     */
    public function create()
    {
        return view('admin.banner.create');
    }

    public function upload(Request $request)
    {
        $this->authorize('create', Banner::class);

        $validator = Validator::make($request->all(), [
            'file' => file_validation(required: true, sometimes: false),
        ]);

        if ($validator->fails()) {
            return response($validator->errors()->first('file'), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        $file = $request->file('file');
        $new_name = $file->hashName();

        $file->storeAs('files/banners', $new_name, 'public');

        $banner = Banner::create([
            'image' => 'files/banners/' . $new_name,
        ]);

        if ($request->ajax()) {
            return response("[SUCCESS] #$banner->id - $new_name");
        } else {
            return redirect()->route('banner.create')->withStatus(__("New banner successfully created."));
        }
    }

    public function reorder(Request $request) {
        $this->authorize('update', Banner::class);

        if ($request->ajax()) {
            foreach ($request->input('rows', []) as $key => $row) {
                Banner::find($row['id'])->update([
                    'position' => $row['position']
                ]);
            }

            return response()->noContent();
        }

        abort(403);
    }

    /**
     * Show the form for editing the specified resource.
     * @param Banner $banner
     * @return Renderable
     */
    public function edit(Banner $banner)
    {
        return view('admin.banner.edit', [
            'banner' => $banner
        ]);
    }

    /**
     * Update the specified resource in storage.
     * @param BannerRequest $request
     * @param Banner $banner
     * @return Renderable
     */
    public function update(BannerRequest $request, Banner $banner)
    {
        $banner->update([
            'title' => $request->get('title'),
            'image' => $request->has('banner') ? (is_string($request->banner) ? $request->banner : $request->banner->store('files/banners', 'public')) : $banner->image,
        ]);

        return redirect()->route('banner.index')->withStatus(__("Banner #$banner->id successfully updated."));
    }

    /**
     * Remove the specified resource from storage.
     * @param Banner $banner
     * @return Renderable
     */
    public function destroy(Banner $banner)
    {
        $banner->delete();

        return redirect()->route('banner.index')->withStatus(__("Banner #$banner->id successfully deleted."));
    }

    public function restore($id)
    {
        $banner = Banner::onlyTrashed()->findOrFail($id);

        $this->authorize('restore', $banner);

        if ($banner->trashed()) {
            $banner->restore();

            return redirect()->route('banner.trashed')->withStatus(__("Banner #$banner->id successfully restored."));
        }

        return redirect()->route('banner.trashed')->withErrors(__("Banner #$banner->id cannot be restored."));
    }
}
