<?php

namespace Modules\User\Entities;

use App\Classes\Dynamics\Models\Customer;
use App\Model\AuditLogin;
use App\Notifications\ResetPassword;
use App\Supports\Address;
use App\Traits\QueryCacheable;
use Deligoez\LaravelModelHashId\Traits\HasHashId;
use Deligoez\LaravelModelHashId\Traits\HasHashIdRouting;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Http\Request;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Carbon;
use Modules\Admin\Entities\Admin;
use Modules\Company\Entities\Company;
use Modules\Country\Entities\Country;
use Modules\User\Database\Factories\UserFactory;
use Modules\User\Notifications\Welcome;
use Yajra\DataTables\Facades\DataTables;

// NOTE: remove "implements MustVerifyEmail" to disable email verification
class User extends Authenticatable implements MustVerifyEmail
{
    use Notifiable;
    use HasFactory;
    use HasHashId;
    use HasHashIdRouting;
    use SoftDeletes;
    use QueryCacheable;

    /**
     * Specify the amount of time to cache queries.
     * Do not specify or set it to null to disable caching.
     *
     * @var int|\DateTime
     */
    public $cacheFor = 3600;

    /**
     * The tags for the query cache. Can be useful
     * if flushing cache for specific tags only.
     *
     * @var null|array
     */
    public $cacheTags = ['users'];

    /**
     * Invalidate the cache automatically
     * upon update in the database.
     *
     * @var bool
     */
    protected static $flushCacheOnUpdate = true;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'picture',
        'gender',
        'email',
        'contact_no',
        'password',
        // 'remember_token',
        'address_line_1',
        'address_line_2',
        'postcode',
        'country_id',
        'country_iso',
        'country_name',
        'is_active',
        // 'last_login_at',
        // 'email_verified_at',
        'deleted_by',
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password',
        'remember_token',
        // 'deleted_at',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'is_active' => 'boolean',
        'updated_at' => 'datetime:Y-m-d H:i:s',
        'created_at' => 'datetime:Y-m-d H:i:s',
        'deleted_at' => 'datetime:Y-m-d H:i:s',
        'email_verified_at' => 'datetime:Y-m-d H:i:s',
        'last_login_at' => 'datetime:Y-m-d H:i:s',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = ['avatar_path'];

    /**
     * Create a new factory instance for the model.
     */
    protected static function newFactory()
    {
        return UserFactory::new();
    }

    public function auditLogins()
    {
        return $this->hasMany(AuditLogin::class);
    }

    public function filteredAuditLogins()
    {
        return $this->auditLogins()->filters(request());
    }

    public function country()
    {
        return $this->belongsTo(Country::class);
    }

    /**
     * Get the path to the profile picture
     *
     * @return string
     */
    public function getAvatarPathAttribute()
    {
        $default = uno()->avatar;

        if ($this->picture) {
            if (filter_var($this->picture, FILTER_VALIDATE_URL)) {
                return $this->picture;
            } else if (asset_exists($this->picture)) {
                return disk_public()->url($this->picture);
            } else {
                return $default;
            }
        }

        return $default;
    }

    protected function address(): Attribute
    {
        return Attribute::make(
            get: fn (mixed $value, array $attributes) => new Address(
                $attributes['address_line_1'] ?: 'N/A',
                $attributes['address_line_2'],
                $attributes['postcode'],
                $attributes['country_name'],
                $attributes['country_id'],
            ),
        );
    }

    /**
     * Send the password reset notification.
     *
     * @param  string  $token
     * @return void
     */
    public function sendPasswordResetNotification($token)
    {
        $this->notify(new ResetPassword($token, 'user.password.reset'));
    }

    public function scopeActive($query)
    {
        $query->where('users.is_active', true);
    }

    public function scopeFilters($query, Request $request, $fav = false)
    {
        $query->when($request->has('country'), function ($query) use ($request) {
            return $query->where('country_id', $request->country);
        });

        $query->when($request->has('status'), function ($query) use ($request) {
            return $query->where('is_active', $request->status == 'active');
        });

        $query->when($request->has('range'), function ($query) use ($request) {
            $date_range = explode(' to ', $request->range);

            if (count($date_range) == 2) {
                $from = new Carbon($date_range[0]);
                $to = new Carbon($date_range[1]);

                $query->whereBetween('created_at', [$from->startOfDay(), $to->endOfDay()]);
            } else if (count($date_range) == 1) {
                $query->whereDate('created_at', $date_range[0]);
            }

            return $query;
        });
    }

    public static function makeDataTable($source, $trash = false, $pricelist = false)
    {
        return DataTables::of($source)
            ->addIndexColumn()
            ->addColumn('avatar', function (self $row) {
                return view('admin.datatables.thumbnail', ['path' => $row->avatar_path]);
            })
            ->addColumn('country', function (self $row) {
                return view('admin.datatables.related', [
                    'row' => $row,
                    'relation' => 'country',
                    // 'route' => 'country.edit',
                    'label' => 'name',
                    'allowed' => false,
                ]);
            })
            ->addColumn('status', function (self $row) {
                return view('admin.datatables.status', ['active' => $row->is_active]);
            })
            ->addColumn('actions', function (self $row) use ($trash) {
                $actions = ['row' => $row];

                if ($trash) {
                    $actions['restore_route'] = 'user.restore';
                } else {
                    $actions['update_route'] = 'user.edit';
                    $actions['delete_route'] = 'user.destroy';
                }

                return view('admin.datatables.actions', $actions);
            })
            // to fixed yajra weird timezone return
            ->editColumn('created_at', fn($row) => $row->created_at)
            ->editColumn('updated_at', fn($row) => $row->updated_at)
            ->editColumn('deleted_at', fn($row) => $row->deleted_at)
            ->make(true);
    }

    public function sendWelcomeNotification()
    {
        return $this->notify(new Welcome());
    }
}
