<?php

namespace Modules\Speaker\Http\Controllers;

use App\Classes\Dynamics\Models\Customer;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Maatwebsite\Excel\Facades\Excel;
use Modules\Company\Entities\Company;
use Modules\Country\Entities\Country;
use Modules\Event\Entities\Event;
use Modules\Speaker\Entities\Speaker;
use Modules\Speaker\Exports\SpeakersExport;
use Modules\Speaker\Http\Requests\SpeakerRequest;
use Modules\Speaker\Notifications\AccountRegistered;
use Str;

class AdminController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(Speaker::class, 'speaker');
    }

    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = Speaker::query()
                // ->with(['country'])
                ->filters($request);

            return Speaker::makeDataTable($data);
        }

        return view('speaker::admin.index', [
            // 'countries' => Country::has('filteredSpeakers')->orderBy('name')->pluck('name', 'id'),
            'events' => Event::has('speakers')->orderBy('title')->pluck('title', 'id'),
        ]);
    }

    public function trashed(Request $request)
    {
        $this->authorize('restore', Speaker::class);

        if ($request->ajax()) {
            $data = Speaker::query()
                // ->with(['country'])
                ->onlyTrashed();

            return Speaker::makeDataTable($data, trash: true);
        }

        return view('speaker::admin.trashed');
    }

    /**
     * Show the form for creating a new resource.
     * @return Renderable
     */
    public function create()
    {
        return view('speaker::admin.create', [
            'speaker' => new Speaker(),
        ]);
    }

    /**
     * Store a newly created speaker in storage
     *
     * @param  \Modules\Speaker\Http\Requests\SpeakerRequest $request
     * @param  \Modules\Speaker\Entities\Speaker $speaker
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(SpeakerRequest $request, Speaker $speaker)
    {
        // $new_password = generate_password();

        /** @var Speaker $new_speaker */
        $new_speaker = $speaker->create(array_merge($request->validated(), [
            'picture' => upload_file($request, 'photo', 'speakers'),
            'is_active' => $request->boolean('is_active'),
            // 'password' => Hash::make($new_password),
            // 'email_verified_at' => now(),
        ]));

        try {
            // $new_speaker->notify(new AccountRegistered($new_password));
        } catch (\Exception $e) {
            report($e);
        }

        return redirect()->route('speaker.index')->withStatus(__('New speaker successfully created.'));
    }

    /**
     * Show the form for editing the specified speaker
     *
     * @param  \Modules\Speaker\Entities\Speaker $speaker
     * @return \Illuminate\View\View
     */
    public function edit(Speaker $speaker)
    {
        return view('speaker::admin.edit', [
            'speaker' => $speaker,
        ]);
    }

    /**
     * Update the specified speaker in storage
     *
     * @param  \Modules\Speaker\Http\Requests\SpeakerRequest $request
     * @param  \Modules\Speaker\Entities\Speaker $speaker
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(SpeakerRequest $request, Speaker $speaker)
    {
        $input = array_merge($request->validated(), [
            'picture' => upload_file($request, 'photo', 'speakers', $speaker->picture),
            'is_active' => $request->boolean('is_active'),
        ]);

        // data_forget($input, 'password');

        $speaker->update($input);

        if ($request->filled('password')) {
            // $speaker->password = Hash::make($request->get('password'));
            // $speaker->save();
        }

        return redirect()->route('speaker.index')->withStatus(__("Speaker #$speaker->id successfully updated."));
    }

    /**
     * Remove the specified speaker from storage
     *
     * @param  \Modules\Speaker\Entities\Speaker $speaker
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Speaker $speaker)
    {
        $speaker->delete();

        return redirect()->route('speaker.index')->withStatus(__("Speaker #$speaker->id successfully deleted."));
    }

    public function restore($id)
    {
        $speaker = Speaker::onlyTrashed()->findOrFail($id);

        $this->authorize('restore', $speaker);

        if ($speaker->trashed()) {
            $speaker->restore();

            return redirect()->route('speaker.trashed')->withStatus(__("Speaker #$speaker->id successfully restored."));
        }

        return redirect()->route('speaker.trashed')->withErrors(__("Speaker #$speaker->id cannot be restored."));
    }

    public function export(Request $request)
    {
        $this->authorize('export', Speaker::class);

        return Excel::download(new SpeakersExport($request), 'speakers-' . now()->format('dmY_His') . '.xlsx');
    }
}
