<?php

namespace Modules\Participant\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\Participant\Entities\Participant;
use Modules\Participant\Http\Requests\ParticipantRequest;
use Maatwebsite\Excel\Facades\Excel;
use Modules\Event\Entities\Event;
use Modules\Participant\Exports\ParticipantsExport;

class AdminController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(Participant::class, 'participant');
    }

    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = Participant::query()
                ->with(['event'])
                ->filters($request);

            return Participant::makeDataTable($data);
        }

        return view('participant::admin.index', [
            'events' => Event::has('participants')->orderBy('title')->pluck('title', 'id'),
        ]);
    }

    public function trashed(Request $request)
    {
        $this->authorize('restore', Participant::class);

        if ($request->ajax()) {
            $data = Participant::query()
                ->with(['event'])
                ->onlyTrashed();

            return Participant::makeDataTable($data, trash: true);
        }

        return view('participant::admin.trashed');
    }

    /**
     * Show the form for creating a new resource.
     * @return Renderable
     */
    public function create()
    {
        return view('participant::admin.create', ['participant' => new Participant()]);
    }

    /**
     * Store a newly created participant in storage
     *
     * @param  \Modules\Participant\Http\Requests\ParticipantRequest $request
     * @param  \Modules\Participant\Entities\Participant $participant
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(ParticipantRequest $request, Participant $participant)
    {
        $participant->create(array_merge($request->validated(), [
            'is_active' => $request->boolean('is_active'),
        ]));

        return redirect()->route('participant.index')->withStatus(__('New participant successfully created.'));
    }

    /**
     * Show the form for editing the specified participant
     *
     * @param  \Modules\Participant\Entities\Participant $participant
     * @return \Illuminate\View\View
     */
    public function edit(Participant $participant)
    {
        return view('participant::admin.edit', ['participant' => $participant]);
    }

    /**
     * Update the specified participant in storage
     *
     * @param  \Modules\Participant\Http\Requests\ParticipantRequest $request
     * @param  \Modules\Participant\Entities\Participant $participant
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(ParticipantRequest $request, Participant $participant)
    {
        $participant->update(array_merge($request->validated(), [
            'is_active' => $request->boolean('is_active'),
        ]));

        return redirect()->route('participant.index')->withStatus(__("Participant #$participant->id successfully updated."));
    }

    /**
     * Remove the specified participant from storage
     *
     * @param  \Modules\Participant\Entities\Participant $participant
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Participant $participant)
    {
        $participant->delete();

        return redirect()->route('participant.index')->withStatus(__("Participant #$participant->id successfully deleted."));
    }

    public function restore($id)
    {
        $participant = Participant::onlyTrashed()->findOrFail($id);

        $this->authorize('restore', $participant);

        if ($participant->trashed()) {
            $participant->restore();

            return redirect()->route('participant.trashed')->withStatus(__("Participant #$participant->id successfully restored."));
        }

        return redirect()->route('participant.trashed')->withErrors(__("Participant #$participant->id cannot be restored."));
    }

    public function export(Request $request)
    {
        $this->authorize('export', Participant::class);

        return Excel::download(new ParticipantsExport($request), 'participants-' . now()->format('dmY_His') . '.xlsx');
    }
}
