<?php

namespace Modules\Feedback\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\Feedback\Entities\Feedback;
use Modules\Feedback\Http\Requests\FeedbackRequest;
use Maatwebsite\Excel\Facades\Excel;
use Modules\Event\Entities\Event;
use Modules\Feedback\Exports\FeedbacksExport;

class AdminController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(Feedback::class, 'feedback');
    }

    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = Feedback::query()
                ->with(['event'])
                ->filters($request);

            return Feedback::makeDataTable($data);
        }

        return view('feedback::admin.index', [
            'events' => Event::has('feedbacks')->orderBy('title')->pluck('title', 'id'),
        ]);
    }

    public function trashed(Request $request)
    {
        $this->authorize('restore', Feedback::class);

        if ($request->ajax()) {
            $data = Feedback::query()
                ->with(['event'])
                ->onlyTrashed();

            return Feedback::makeDataTable($data, trash: true);
        }

        return view('feedback::admin.trashed');
    }

    /**
     * Show the form for creating a new resource.
     * @return Renderable
     */
    public function create()
    {
        return view('feedback::admin.create', [
            'feedback' => new Feedback(),
            'events' => Event::active()->orderBy('title')->pluck('title', 'id'),
        ]);
    }

    /**
     * Store a newly created feedback in storage
     *
     * @param  \Modules\Feedback\Http\Requests\FeedbackRequest $request
     * @param  \Modules\Feedback\Entities\Feedback $feedback
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(FeedbackRequest $request, Feedback $feedback)
    {
        $feedback->create(array_merge($request->validated(), [
            // 'is_anonymous' => $request->boolean('is_anonymous'),
        ]));

        return redirect()->route('feedback.index')->withStatus(__('New feedback successfully created.'));
    }

    /**
     * Show the form for editing the specified feedback
     *
     * @param  \Modules\Feedback\Entities\Feedback $feedback
     * @return \Illuminate\View\View
     */
    public function edit(Feedback $feedback)
    {
        return view('feedback::admin.edit', [
            'feedback' => $feedback,
            'events' => Event::has('feedbacks')->orderBy('title')->pluck('title', 'id'),
        ]);
    }

    /**
     * Update the specified feedback in storage
     *
     * @param  \Modules\Feedback\Http\Requests\FeedbackRequest $request
     * @param  \Modules\Feedback\Entities\Feedback $feedback
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(FeedbackRequest $request, Feedback $feedback)
    {
        $feedback->update(array_merge($request->validated(), [
            // 'is_anonymous' => $request->boolean('is_anonymous'),
        ]));

        return redirect()->route('feedback.index')->withStatus(__("Feedback #$feedback->id successfully updated."));
    }

    /**
     * Remove the specified feedback from storage
     *
     * @param  \Modules\Feedback\Entities\Feedback $feedback
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Feedback $feedback)
    {
        $feedback->delete();

        return redirect()->route('feedback.index')->withStatus(__("Feedback #$feedback->id successfully deleted."));
    }

    public function restore($id)
    {
        $feedback = Feedback::onlyTrashed()->findOrFail($id);

        $this->authorize('restore', $feedback);

        if ($feedback->trashed()) {
            $feedback->restore();

            return redirect()->route('feedback.trashed')->withStatus(__("Feedback #$feedback->id successfully restored."));
        }

        return redirect()->route('feedback.trashed')->withErrors(__("Feedback #$feedback->id cannot be restored."));
    }

    public function export(Request $request)
    {
        $this->authorize('export', Feedback::class);

        return Excel::download(new FeedbacksExport($request), 'feedbacks-' . now()->format('dmY_His') . '.xlsx');
    }
}
