<?php

namespace Modules\Event\Entities;

use App\Traits\QueryCacheable;
use Deligoez\LaravelModelHashId\Traits\HasHashId;
use Deligoez\LaravelModelHashId\Traits\HasHashIdRouting;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Modules\Admin\Entities\Admin;
use Modules\Event\Database\Factories\EventScheduleFactory;
use Modules\Speaker\Entities\Speaker;

class EventSchedule extends Model
{
    use HasFactory;
    use HasHashId;
    use HasHashIdRouting;
    use QueryCacheable;

    const TAG_INVITE = "invite";
    const TAG_PUBLIC = "public";

    /**
     * Specify the amount of time to cache queries.
     * Do not specify or set it to null to disable caching.
     *
     * @var int|\DateTime
     */
    public $cacheFor = 3600;

    /**
     * The tags for the query cache. Can be useful
     * if flushing cache for specific tags only.
     *
     * @var null|array
     */
    public $cacheTags = ['event_schedules'];

    /**
     * Invalidate the cache automatically
     * upon update in the database.
     *
     * @var bool
     */
    protected static $flushCacheOnUpdate = true;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'event_id',
        'position',
        'title',
        'description',
        'tags',
        'venue',
        'date',
        'start_time',
        'end_time',
        'is_active',
        'created_by',
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        //
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'is_active' => 'boolean',
        'updated_at' => 'datetime:Y-m-d H:i:s',
        'created_at' => 'datetime:Y-m-d H:i:s',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = [
        //
    ];

    protected static function boot()
    {
        parent::boot();

        self::creating(function ($model) {
            $model->position = self::where('event_id', $model->event_id)->max('position') + 1;
        });

        self::deleted(function ($model) {
            //
        });
    }

    /**
     * Create a new factory instance for the model.
     */
    protected static function newFactory()
    {
        return EventScheduleFactory::new();
    }

    public function event()
    {
        return $this->belongsTo(Event::class);
    }

    public function creator()
    {
        return $this->belongsTo(Admin::class, 'created_by');
    }

    public function speakers()
    {
        return $this->belongsToMany(Speaker::class, 'schedule_speaker', 'schedule_id')->orderBy('name');
    }

    public function activeSpeakers()
    {
        return $this->speakers()->active();
    }

    public function scopeActive($query)
    {
        $query->where('event_schedules.is_active', true);
    }

    public function getFullTimeAttribute()
    {
        return time_formatted($this->start_time, 'H:i') . ' - ' . time_formatted($this->end_time, 'H:i');
    }

    public static function listTags()
    {
        return [
            self::TAG_INVITE => __('By invite only'),
            self::TAG_PUBLIC => __('Open to public'),
        ];
    }
}
