<?php

namespace Modules\Event\Entities;

use App\Traits\QueryCacheable;
use Deligoez\LaravelModelHashId\Traits\HasHashId;
use Deligoez\LaravelModelHashId\Traits\HasHashIdRouting;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Modules\Admin\Entities\Admin;
use Modules\Event\Database\Factories\EventPhotoFactory;
use Nette\Utils\Html;
use Yajra\DataTables\Facades\DataTables;

class EventPhoto extends Model
{
    const TYPE_PHOTOS = 'photos';
    const TYPE_VIDEOS = 'videos';
    const TYPE_UNKNOWN = 'unknown';

    use HasFactory;
    use HasHashId;
    use HasHashIdRouting;
    use QueryCacheable;

    /**
     * Specify the amount of time to cache queries.
     * Do not specify or set it to null to disable caching.
     *
     * @var int|\DateTime
     */
    public $cacheFor = 3600;

    /**
     * The tags for the query cache. Can be useful
     * if flushing cache for specific tags only.
     *
     * @var null|array
     */
    public $cacheTags = ['event_photos'];

    /**
     * Invalidate the cache automatically
     * upon update in the database.
     *
     * @var bool
     */
    protected static $flushCacheOnUpdate = true;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'event_id',
        'position',
        'thumbnail',
        'path',
        'name',
        'extension',
        'mime',
        'size',
        'is_active',
        'created_by',
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        // 'deleted_at',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'is_active' => 'boolean',
        'updated_at' => 'datetime:Y-m-d H:i:s',
        'created_at' => 'datetime:Y-m-d H:i:s',
        'deleted_at' => 'datetime:Y-m-d H:i:s',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = [
        //
    ];

    protected static function boot()
    {
        parent::boot();

        self::creating(function ($model) {
            $model->position = self::where('event_id', $model->event_id)->max('position') + 1;
        });

        self::deleted(function ($model) {
            destroy_asset($model->thumbnail);
            destroy_asset($model->path);
        });
    }

    /**
     * Create a new factory instance for the model.
     */
    protected static function newFactory()
    {
        return EventPhotoFactory::new();
    }

    public function event()
    {
        return $this->belongsTo(Event::class);
    }

    public function creator()
    {
        return $this->belongsTo(Admin::class, 'created_by');
    }

    public function scopeActive($query)
    {
        $query->where('event_photos.is_active', true);
    }

    public function getThumbnailPathAttribute()
    {
        if (filter_var($this->thumbnail, FILTER_VALIDATE_URL)) {
            return $this->thumbnail;
        } else if (asset_exists($this->thumbnail)) {
            return disk_public()->url($this->thumbnail);
        } else if ((api_request() || route_match('event.gallery.index')) && $this->type == self::TYPE_PHOTOS) {
            return $this->full_path;
        } else if ($this->type == self::TYPE_VIDEOS) {
            return uno()->no_video;
        }

        return uno()->no_image;
    }

    public function getFullPathThumbnailAttribute()
    {
        $default = uno()->no_image;

        if ($this->type == self::TYPE_PHOTOS) {
            return $this->full_path ?: uno()->no_image;
        } else if ($this->type == self::TYPE_VIDEOS) {
            $fallback = "images/icons/$this->extension-fallback.svg";
            return asset_exists($fallback) ? asset($fallback) : uno()->no_video;
        }

        return $default;
    }

    public function getFullPathAttribute()
    {
        if (filter_var($this->path, FILTER_VALIDATE_URL)) {
            return $this->path;
        } else if (asset_exists($this->path)) {
            return disk_public()->url($this->path);
        }

        return null;
    }

    public function getTypeAttribute()
    {
        if ($this->path) {
            if (str_starts_with($this->mime, 'image/')) {
                return self::TYPE_PHOTOS;
            } else if (in_array($this->extension, ['mov', 'mp4'])) {
                return self::TYPE_VIDEOS;
            }
        }

        return self::TYPE_UNKNOWN;
    }

    public function scopeFilters($query, Request $request)
    {
        $query->when($request->has('status'), function ($query) use ($request) {
            return $query->where('is_active', $request->status == 'active');
        });

        $query->when($request->has('type'), function ($query) use ($request) {
            return $query->whereLike('mime', $request->type . '%');
        });

        $query->when($request->has('range'), function ($query) use ($request) {
            $date_range = explode(' to ', $request->range);

            if (count($date_range) == 2) {
                $from = new Carbon($date_range[0]);
                $to = new Carbon($date_range[1]);

                $query->whereBetween('created_at', [$from->startOfDay(), $to->endOfDay()]);
            } else if (count($date_range) == 1) {
                $query->whereDate('created_at', $date_range[0]);
            }

            return $query;
        });
    }

    public static function makeDataTable($source, $belongs = null)
    {
        return DataTables::of($source)
            ->addIndexColumn()
            ->editColumn('thumbnail', function (self $row) {
                return view('admin.datatables.thumbnail', ['path' => $row->thumbnail_path]);
            })
            ->addColumn('file', function (self $row) {
                return Html::el('a', [
                        'href' => $url = $row->full_path,
                        'data-placement' => 'right',
                        'data-toggle' => 'tooltip',
                        'title' => $url,
                        'target' => '_blank',
                    ])
                    ->addHtml(Html::el('span', ['class' => 'd-inline-block mr-2'])->addText($row->name))
                    ->addHtml(Html::el('i class="fas fa-arrow-up-right-from-square"'));
            })
            ->addColumn('type', function (self $row) {
                return view('admin.datatables.status', [
                    'active' => str_starts_with($row->mime, 'image/'),
                    'true' => __('Photo'),
                    'false' => __('Video'),
                ]);
            })
            ->editColumn('size', function (self $row) {
                return format_bytes($row->size);
            })
            ->addColumn('actions', function (self $row) use ($belongs) {
                $actions = [
                    'row' => $row,
                    'params' => [$belongs, $row],
                    'allowed_update' => true,
                    'update_route' => 'event.gallery.edit',
                    'allowed_delete' => true,
                    'delete_route'=> 'event.gallery.destroy',
                ];

                return view('admin.datatables.actions', $actions);
            })
            // to fixed yajra weird timezone return
            ->editColumn('created_at', fn($row) => $row->created_at)
            ->editColumn('updated_at', fn($row) => $row->updated_at)
            ->editColumn('deleted_at', fn($row) => $row->deleted_at)
            ->make(true);
    }
}
