<?php

namespace Modules\Event\Entities;

use App\Traits\QueryCacheable;
use Deligoez\LaravelModelHashId\Traits\HasHashId;
use Deligoez\LaravelModelHashId\Traits\HasHashIdRouting;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Modules\Admin\Entities\Admin;
use Modules\Country\Entities\Country;
use Modules\Event\Database\Factories\EventFactory;
use Modules\Feedback\Entities\Feedback;
use Modules\Participant\Entities\Participant;
use Modules\Speaker\Entities\Speaker;
use Nette\Utils\Html;
use Yajra\DataTables\Facades\DataTables;

class Event extends Model
{
    const LABEL_PAST = 'past';
    const LABEL_ONGOING = 'ongoing';
    const LABEL_UPCOMING = 'upcoming';

    use HasFactory;
    use HasHashId;
    use HasHashIdRouting;
    use SoftDeletes;
    use QueryCacheable;

    /**
     * Specify the amount of time to cache queries.
     * Do not specify or set it to null to disable caching.
     *
     * @var int|\DateTime
     */
    public $cacheFor = 3600;

    /**
     * The tags for the query cache. Can be useful
     * if flushing cache for specific tags only.
     *
     * @var null|array
     */
    public $cacheTags = ['events'];

    /**
     * Invalidate the cache automatically
     * upon update in the database.
     *
     * @var bool
     */
    protected static $flushCacheOnUpdate = true;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'title',
        'slug',
        'short_description',
        'description',
        'location',
        'timezone',
        'address_line_1',
        'address_line_2',
        'postcode',
        'country_id',
        'country_iso',
        'country_name',
        'thumbnail',
        'banner',
        'file_path',
        'file_name',
        'start_date',
        'end_date',
        'no_expiry',
        'is_featured',
        'is_active',
        'enable_submit_feedback',
        'created_by',
        'deleted_by',
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        // 'deleted_at',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'no_expiry' => 'boolean',
        'is_featured' => 'boolean',
        'is_active' => 'boolean',
        'enable_submit_feedback' => 'boolean',
        'start_date' => 'datetime:Y-m-d',
        'end_date' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d H:i:s',
        'created_at' => 'datetime:Y-m-d H:i:s',
        'deleted_at' => 'datetime:Y-m-d H:i:s',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = [
        //
    ];

    /**
     * Create a new factory instance for the model.
     */
    protected static function newFactory()
    {
        return EventFactory::new();
    }

    public function country()
    {
        return $this->belongsTo(Country::class);
    }

    public function creator()
    {
        return $this->belongsTo(Admin::class, 'created_by');
    }

    public function deletor()
    {
        return $this->belongsTo(Admin::class, 'deleted_by');
    }

    public function speakers()
    {
        return $this->belongsToMany(Speaker::class)->orderBy('name');
    }

    public function activeSpeakers()
    {
        return $this->speakers()->active();
    }

    public function photos()
    {
        return $this->hasMany(EventPhoto::class, 'event_id');
    }

    public function activePhotos()
    {
        return $this->photos()->active();
    }

    public function participants()
    {
        return $this->hasMany(Participant::class, 'event_id');
    }

    public function feedbacks()
    {
        return $this->hasMany(Feedback::class, 'event_id');
    }

    public function relatedEvents()
    {
        return $this->belongsToMany(
            Event::class,
            'related_events',
            'event_id',           // this model's key
            'related_event_id'    // related model's key
        )->orderBy('title');
    }

    public function activeRelatedEvents()
    {
        return $this->relatedEvents()->active();
    }
    public function pendingFeedbacks()
    {
        return $this->feedbacks()->pending();
    }

    public function approvedFeedbacks()
    {
        return $this->feedbacks()->approved();
    }

    public function disabledFeedbacks()
    {
        return $this->feedbacks()->disabled();
    }

    public function schedules()
    {
        return $this->hasMany(EventSchedule::class, 'event_id')
            ->orderBy('date')->orderBy('position');
    }

    public function activeSchedules()
    {
        return $this->schedules()->active();
    }

    public function scopeApi($query)
    {
        $query->with(['activeSpeakers', 'activeSchedules', 'activePhotos', 'approvedFeedbacks' , 'activeRelatedEvents']);
    }

    public function scopeActive($query)
    {
        $query->where('events.is_active', true);
    }

    public function scopeInactive($query)
    {
        $query->where('events.is_active', false);
    }

    public function scopeFeatured($query)
    {
        $query->where('events.is_featured', true);
    }

    public function scopeFilters($query, Request $request)
    {
        $query->when($request->has('status'), function ($query) use ($request) {
            return $query->where('is_active', $request->status == 'active');
        });

        $query->when($request->has('featured'), function ($query) use ($request) {
            return $query->where('is_featured', $request->featured == 'yes');
        });

        $query->when($request->has('feedback'), function ($query) use ($request) {
            return $query->where('enable_submit_feedback', $request->feedback == 'enabled');
        });

        $query->when($request->has('speaker'), function ($query) use ($request) {
            return $query->whereRelation('speakers', 'id', $request->speaker);
        });

        $query->when($request->has('year'), function ($query) use ($request) {
            return $query->whereYear('start_date', $request->year);
        });

        $query->when($request->has('keyword'), function ($query) use ($request) {
            return $query->where(function ($q) use ($request) {
                $q->whereLike('title', '%' . $request->keyword . '%');
                $q->orWhereLike('location', '%' . $request->keyword . '%');
                $q->orWhereLike('description', '%' . $request->keyword . '%');
            });
        });

        $query->when($request->has('range'), function ($query) use ($request) {
            $date_range = explode(' to ', $request->range);

            if (count($date_range) == 2) {
                $from = new Carbon($date_range[0]);
                $to = new Carbon($date_range[1]);

                $query->whereBetween('created_at', [$from->startOfDay(), $to->endOfDay()]);
            } else if (count($date_range) == 1) {
                $query->whereDate('created_at', $date_range[0]);
            }

            return $query;
        });
    }

    public function getThumbnailPathAttribute()
    {
        $default = uno()->no_image;

        if ($this->thumbnail) {
            if (filter_var($this->thumbnail, FILTER_VALIDATE_URL)) {
                return $this->thumbnail;
            } else if (asset_exists($this->thumbnail)) {
                return disk_public()->url($this->thumbnail);
            } else {
                return $default;
            }
        }

        return $default;
    }

    public function getBannerPathAttribute()
    {
        $default = uno()->no_image;

        if ($this->banner) {
            if (filter_var($this->banner, FILTER_VALIDATE_URL)) {
                return $this->banner;
            } else if (asset_exists($this->banner)) {
                return disk_public()->url($this->banner);
            } else {
                return $default;
            }
        }

        return $default;
    }

    public function getFileUrlAttribute()
    {
        if ($this->file_path && asset_exists($this->file_path)) {
            return disk_public()->url($this->file_path);
        }

        return null;
    }

    public function getFileDownloadUrlAttribute()
    {
        if ($this->file_path && asset_exists($this->file_path)) {
            return route('front.event.download', $this);
        }

        return null;
    }

    public function getFileThumbnailUrlAttribute()
    {
        if ($this->file_url) {
            $fallback = "images/icons/pdf-fallback.svg";

            if (asset_exists($fallback)) {
                return asset($fallback);
            }
        }

        return uno()->no_image;
    }

    public function getLabelAttribute()
    {
        if ((!is_null($this->end_date) && $this->end_date->lt(today())) || ($this->start_date->lt(today()) && is_null($this->end_date))) {
            return self::LABEL_PAST;
        } else if ($this->start_date->lte(today()) && (is_null($this->end_date) || $this->end_date->gte(today()))) {
            return self::LABEL_ONGOING;
        } else {
            return self::LABEL_UPCOMING;
        }
    }

    public static function makeDataTable($source, $trash = false)
    {
        return DataTables::of($source)
            ->addIndexColumn()
            // ->addColumn('speakers', function (self $row) {
            //     return view('admin.datatables.multirelated', [
            //         'row' => $row,
            //         'relation' => 'speakers',
            //         'route' => 'speaker.edit',
            //         'label' => 'name'
            //     ]);
            // })
            ->editColumn('photos_count', function (self $row) {
                return $row->photos_count ? view('admin.datatables.open', [
                    'entity' => EventPhoto::class,
                    'route' => 'event.gallery.index',
                    'params' => $row,
                    'label' => $row->photos_count,
                    'allowed' => true,
                ]) : 0;
            })
            ->editColumn('speakers_count', function (self $row) {
                return $row->speakers_count ? view('admin.datatables.open', [
                    'entity' => Speaker::class,
                    'route' => 'speaker.index',
                    'params' => ['event' => $row->id],
                    'label' => $row->speakers_count
                ]) : 0;
            })
            ->editColumn('participants_count', function (self $row) {
                return $row->participants_count ? view('admin.datatables.open', [
                    'entity' => Participant::class,
                    'route' => 'participant.index',
                    'params' => ['event' => $row->id],
                    'label' => $row->participants_count
                ]) : 0;
            })
            ->editColumn('feedbacks_count', function (self $row) {
                return $row->feedbacks_count ? view('admin.datatables.open', [
                    'entity' => Feedback::class,
                    'route' => 'feedback.index',
                    'params' => ['event' => $row->id],
                    'label' => $row->feedbacks_count
                ]) : 0;
            })
            ->editColumn('thumbnail', function (self $row) {
                return view('admin.datatables.thumbnail', ['path' => $row->thumbnail_path]);
            })
            ->editColumn('banner', function (self $row) {
                return view('admin.datatables.thumbnail', ['path' => $row->banner_path]);
            })
            ->editColumn('description', function (self $row) {
                return view('admin.datatables.limit', ['title' => html_entity_decode(strip_tags($row->description)), 'limit' => 40]);
            })
            ->addColumn('feature', function (self $row) {
                return view('admin.datatables.status', ['active' => $row->is_featured, 'true' => __('Yes'), 'false' => __('No')]);
            })
            ->addColumn('feedback', function (self $row) {
                return view('admin.datatables.status', ['active' => $row->enable_submit_feedback, 'true' => __('Enabled'), 'false' => __('Disabled')]);
            })
            ->addColumn('status', function (self $row) {
                return view('admin.datatables.status', ['active' => $row->is_active]);
            })
            ->editColumn('file', function (self $row) {
                return $row->file_path && $row->file_name ? Html::el('a', [
                    'href' => $url = $row->file_url,
                    'data-placement' => 'right',
                    'data-toggle' => 'tooltip',
                    'title' => $url,
                    'target' => '_blank',
                ])
                    ->addHtml(Html::el('span', ['class' => 'd-inline-block mr-2'])->addText($row->file_name))
                    ->addHtml(Html::el('i class="fas fa-arrow-up-right-from-square"')) : null;
            })
            ->addColumn('actions', function (self $row) use ($trash) {
                $actions = ['row' => $row];

                if ($trash) {
                    $actions['restore_route'] = 'event.restore';
                } else {
                    $actions['update_route'] = 'event.edit';
                    // $actions['gallery_route'] = 'event.gallery.index';
                    $actions['schedule_route'] = 'event.schedule.index';
                    // $actions['participants_route'] = 'event.users.index';
                    $actions['delete_route'] = 'event.destroy';
                }

                return view('admin.datatables.actions', $actions);
            })
            // to fixed yajra weird timezone return
            ->editColumn('created_at', fn($row) => $row->created_at)
            ->editColumn('updated_at', fn($row) => $row->updated_at)
            ->editColumn('deleted_at', fn($row) => $row->deleted_at)
            ->make(true);
    }

    public function dateRange($empty = false)
    {
        $dates = collect();

        if (is_null($this->end_date)) {
            return $dates->put($this->start_date->format('Y-m-d'), $empty ? collect() : $this->start_date->format('d F Y'));
        }

        for ($date = $this->start_date->startOfDay(); $date->lte($this->end_date->startOfDay()); $date->addDay()) {
            $dates->put($date->format('Y-m-d'), $empty ? collect() : $date->format('d F Y'));
        }

        return $dates;
    }
}
