<?php

namespace Modules\Event\Database\Factories;

use DateInterval;
use DatePeriod;
use DateTime;
use Illuminate\Database\Eloquent\Factories\Factory;
use Modules\Event\Entities\Event;
use Modules\Event\Entities\EventPhoto;
use Modules\Feedback\Entities\Feedback;
use Modules\Participant\Entities\Participant;
use Modules\Speaker\Entities\Speaker;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Model>
 */
class EventFactory extends Factory
{
    /**
     * The name of the factory's corresponding model.
     *
     * @var string
     */
    protected $model = \Modules\Event\Entities\Event::class;

    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition()
    {
        return [
            'title' => $title = str(fake()->realTextBetween(30, 40))->slug()->headline(),
            'slug' => str($title)->slug(),
            'description' => fake()->realText(),
            'location' => fake()->streetAddress(),
            'banner' => fallback_via_placeholder(fake()->imageUrl(category: $title, randomize: false)),
            'file_path' => 'files/sample/sample.pdf',
            'file_name' => str(fake()->words(asText: true))->slug() . '.pdf',
            'start_date' => $start = fake()->dateTimeBetween('now', '+1 month'),
            'end_date' => carbon($start)->addDays(rand(1, 4))->format('Y-m-d'),
            'is_featured' => fake()->boolean(80),
            'is_active' => fake()->boolean(60),
            'created_by' => 1,
        ];
    }

    public function relationships(): static
    {
        return $this->afterCreating(function (Event $event) {
            $this->speakers($event);
            $this->photos($event);
            $this->feedbacks($event);
            $this->participants($event);
            $this->schedules($event);
        });
    }

    public function speakers(Event $event)
    {
        $event->speakers()->attach(Speaker::inRandomOrder()->take(rand(1, 5))->pluck('id'));
    }

    public function photos(Event $event)
    {
        EventPhoto::factory(rand(5, 15))->create(['event_id' => $event->id]);
    }

    public function feedbacks(Event $event)
    {
        Feedback::factory(rand(5, 15))->create(['event_id' => $event->id]);
    }

    public function participants(Event $event)
    {
        Participant::factory(rand(5, 15))->create(['event_id' => $event->id]);
    }

    public function schedules(Event $event)
    {
        foreach ($event->dateRange()->keys() as $date) {
            for ($i = 0; $i < rand(1, 2); $i++) {
                $random_hour = rand(7, 19);
                $random_minute = fake()->randomElement([0, 15, 30, 45]);
                $random_time = today()->setTime($random_hour, $random_minute, 0);

                $schedule = $event->schedules()->create([
                    'date' => $date,
                    'title' => $title = str(fake()->realTextBetween(30, 40))->slug()->headline(),
                    'venue' => fake()->streetName(),
                    'start_time' => $random_time->clone()->format('H:i'),
                    'end_time' => $random_time->clone()->addHours(rand(1, 2))->format('H:i'),
                    'created_by' => 1,
                ]);

                $speakers = $event->speakers()->inRandomOrder()->take(rand(1, 2))->pluck('id');

                $schedule->speakers()->attach($speakers);
            }
        }
    }
}
