<?php

namespace Modules\Country\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\Country\Entities\Country;
use Modules\Country\Http\Requests\CountryRequest;
use Maatwebsite\Excel\Facades\Excel;
use Modules\Country\Exports\CountriesExport;

class AdminController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(Country::class, 'country');
    }

    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = Country::query()
                ->filters($request);

            return Country::makeDataTable($data);
        }

        return view('country::index');
    }

    public function trashed(Request $request)
    {
        $this->authorize('restore', Country::class);

        if ($request->ajax()) {
            $data = Country::onlyTrashed();

            return Country::makeDataTable($data, trash: true);
        }

        return view('country::trashed');
    }

    /**
     * Show the form for creating a new resource.
     * @return Renderable
     */
    public function create()
    {
        return view('country::create', ['country' => new Country()]);
    }

    /**
     * Store a newly created country in storage
     *
     * @param  \Modules\Country\Http\Requests\CountryRequest $request
     * @param  \Modules\Country\Entities\Country $country
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(CountryRequest $request, Country $country)
    {
        $country->create($request->validated());

        return redirect()->route('country.index')->withStatus(__('New country successfully created.'));
    }

    /**
     * Show the form for editing the specified country
     *
     * @param  \Modules\Country\Entities\Country $country
     * @return \Illuminate\View\View
     */
    public function edit(Country $country)
    {
        return view('country::edit', ['country' => $country]);
    }

    /**
     * Update the specified country in storage
     *
     * @param  \Modules\Country\Http\Requests\CountryRequest $request
     * @param  \Modules\Country\Entities\Country $country
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(CountryRequest $request, Country $country)
    {
        $country->update($request->validated());

        return redirect()->route('country.index')->withStatus(__("Country #$country->id successfully updated."));
    }

    /**
     * Remove the specified country from storage
     *
     * @param  \Modules\Country\Entities\Country $country
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Country $country)
    {
        $country->delete();

        return redirect()->route('country.index')->withStatus(__("Country #$country->id successfully deleted."));
    }

    public function restore($id)
    {
        $country = Country::onlyTrashed()->findOrFail($id);

        $this->authorize('restore', $country);

        if ($country->trashed()) {
            $country->restore();

            return redirect()->route('country.trashed')->withStatus(__("Country #$country->id successfully restored."));
        }

        return redirect()->route('country.trashed')->withErrors(__("Country #$country->id cannot be restored."));
    }

    public function export(Request $request)
    {
        $this->authorize('export', Country::class);

        return Excel::download(new CountriesExport($request), 'countries-' . now()->format('dmY_His') . '.xlsx');
    }
}
