<?php

namespace Modules\Category\Entities;

use App\Traits\QueryCacheable;
use Deligoez\LaravelModelHashId\Traits\HasHashId;
use Deligoez\LaravelModelHashId\Traits\HasHashIdRouting;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Modules\Admin\Entities\Admin;
use Modules\Category\Database\Factories\CategoryFactory;
use Modules\Document\Entities\Document;
use Modules\Marketing\Entities\Marketing;
use Modules\Training\Entities\Training;
use Yajra\DataTables\Facades\DataTables;

class Category extends Model
{
    use HasFactory;
    use HasHashId;
    use HasHashIdRouting;
    use SoftDeletes;
    use QueryCacheable;

    /**
     * Specify the amount of time to cache queries.
     * Do not specify or set it to null to disable caching.
     *
     * @var int|\DateTime
     */
    public $cacheFor = 3600;

    /**
     * The tags for the query cache. Can be useful
     * if flushing cache for specific tags only.
     *
     * @var null|array
     */
    public $cacheTags = ['categories'];

    /**
     * Invalidate the cache automatically
     * upon update in the database.
     *
     * @var bool
     */
    protected static $flushCacheOnUpdate = true;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'parent_id',
        'entity_type',
        'entity_id',
        'position',
        'name',
        'slug',
        'description',
        'keywords',
        'is_active',
        'created_by',
        // 'deleted_by',
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        // 'deleted_at',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'is_active' => 'boolean',
        'updated_at' => 'datetime:Y-m-d H:i:s',
        'created_at' => 'datetime:Y-m-d H:i:s',
        'deleted_at' => 'datetime:Y-m-d H:i:s',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = [
        // 'parent_is_active',
    ];

    /**
     * Create a new factory instance for the model.
     */
    protected static function newFactory()
    {
        return CategoryFactory::new();
    }

    public function parent()
    {
        return $this->belongsTo(self::class, 'parent_id');
    }

    public function activeParent()
    {
        return $this->parent()->active();
    }

    public function children()
    {
        return $this->hasMany(self::class, 'parent_id')->orderBy('position');
    }

    public function activeChildren()
    {
        return $this->children()->active();
    }

    public function filteredChildren()
    {
        return $this->children()->filters(request());
    }

    public function marketings()
    {
        return $this->hasMany(Marketing::class);
    }

    public function activeMarketings()
    {
        return $this->marketings()->active();
    }

    public function trainings()
    {
        return $this->hasMany(Training::class);
    }

    public function activeTrainings()
    {
        return $this->trainings()->active();
    }

    public function documents()
    {
        return $this->hasMany(Document::class);
    }

    public function activeDocuments()
    {
        return $this->documents()->active();
    }

    public function creator()
    {
        return $this->belongsTo(Admin::class, 'created_by');
    }

    public function deleter()
    {
        return $this->belongsTo(Admin::class, 'created_by');
    }

    public function scopeActive($query)
    {
        $query->where('categories.is_active', true);
    }

    public function scopeMain($query)
    {
        $query->whereNull('categories.parent_id');
    }

    public function scopeTypeMarketing($query)
    {
        $query->where('entity_type', Marketing::class);
    }

    public function scopeTypeTraining($query)
    {
        $query->where('entity_type', Training::class);
    }

    public function scopeTypeDocument($query)
    {
        $query->where('entity_type', Document::class);
    }

    public function scopeFilters($query, Request $request)
    {
        $query->when($request->has('group'), function ($query) use ($request) {
            return $query->where('parent_id', $request->group);
        });

        $query->when($request->has('status'), function ($query) use ($request) {
            return $query->where('is_active', $request->status == 'active');
        });

        $query->when($request->has('range'), function ($query) use ($request) {
            $date_range = explode(' to ', $request->range);

            if (count($date_range) == 2) {
                $from = new Carbon($date_range[0]);
                $to = new Carbon($date_range[1]);

                $query->whereBetween('created_at', [$from->startOfDay(), $to->endOfDay()]);
            } else if (count($date_range) == 1) {
                $query->whereDate('created_at', $date_range[0]);
            }

            return $query;
        });
    }

    public static function makeDataTable($source, $trash = false)
    {
        return DataTables::of($source)
            // ->editColumn('keywords', function (self $row) {
            //     return view('admin.datatables.limit', ['title' => $row->keywords, 'limit' => 100]);
            // })
            ->addColumn('status', function (self $row) {
                return view('admin.datatables.status', ['active' => $row->is_active]);
            })
            ->addColumn('actions', function (self $row) use ($trash) {
                $actions = ['row' => $row];

                if ($trash) {
                    $actions['restore_route'] = 'category.restore';
                } else {
                    $actions['update_route'] = 'category.edit';
                    $actions['delete_route'] = 'category.destroy';
                }

                return view('admin.datatables.actions', $actions);
            })
            // to fixed yajra weird timezone return
            ->editColumn('created_at', fn($row) => $row->created_at)
            ->editColumn('updated_at', fn($row) => $row->updated_at)
            ->editColumn('deleted_at', fn($row) => $row->deleted_at)
            ->make(true);
    }

    public function getParentIsActiveAttribute()
    {
        if (is_null($this->parent_id)) {
            return $this->is_active;
        }

        $this->loadMissing('parent');

        return $this->parent->is_active;
    }
}
