<?php

namespace Modules\Block\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\Block\Entities\Block;
use Modules\Block\Http\Requests\BlockRequest;
use Maatwebsite\Excel\Facades\Excel;
use Modules\Block\Exports\BlocksExport;

class AdminController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(Block::class, 'block');
    }

    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = Block::query()
                ->filters($request);

            return Block::makeDataTable($data);
        }

        return view('block::index');
    }

    public function trashed(Request $request)
    {
        $this->authorize('restore', Block::class);

        if ($request->ajax()) {
            $data = Block::query()
                ->onlyTrashed();

            return Block::makeDataTable($data, trash: true);
        }

        return view('block::trashed');
    }

    /**
     * Show the form for creating a new resource.
     * @return Renderable
     */
    public function create()
    {
        return view('block::create', ['block' => new Block()]);
    }

    /**
     * Store a newly created block in storage
     *
     * @param  \Modules\Block\Http\Requests\BlockRequest $request
     * @param  \Modules\Block\Entities\Block $block
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(BlockRequest $request, Block $block)
    {
        $block->create(array_merge($request->validated(), [
            'editor_content' => remove_break_new_lines(str_replace(['<p></p>', '<div></div>'], '<br>', $content = $request->input('editor_content'))),
            'plain_content' => strlen($content) ? html_entity_decode(strip_tags($content)) : '',
            'is_active' => $request->boolean('is_active'),
        ]));

        return redirect()->route('block.index')->withStatus(__('New block successfully created.'));
    }

    /**
     * Show the form for editing the specified block
     *
     * @param  \Modules\Block\Entities\Block $block
     * @return \Illuminate\View\View
     */
    public function edit(Block $block)
    {
        return view('block::edit', ['block' => $block]);
    }

    /**
     * Update the specified block in storage
     *
     * @param  \Modules\Block\Http\Requests\BlockRequest $request
     * @param  \Modules\Block\Entities\Block $block
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(BlockRequest $request, Block $block)
    {
        $block->update(array_merge($request->validated(), [
            'editor_content' => remove_break_new_lines(str_replace(['<p></p>', '<div></div>'], '<br>', $content = $request->input('editor_content'))),
            'plain_content' => strlen($content) ? html_entity_decode(strip_tags($content)) : '',
            'is_active' => $request->boolean('is_active'),
        ]));

        return redirect()->route('block.index')->withStatus(__("Block #$block->id successfully updated."));
    }

    /**
     * Remove the specified block from storage
     *
     * @param  \Modules\Block\Entities\Block $block
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Block $block)
    {
        $block->delete();

        return redirect()->route('block.index')->withStatus(__("Block #$block->id successfully deleted."));
    }

    public function restore($id)
    {
        $block = Block::onlyTrashed()->findOrFail($id);

        $this->authorize('restore', $block);

        if ($block->trashed()) {
            $block->restore();

            return redirect()->route('block.trashed')->withStatus(__("Block #$block->id successfully restored."));
        }

        return redirect()->route('block.trashed')->withErrors(__("Block #$block->id cannot be restored."));
    }

    public function export(Request $request)
    {
        $this->authorize('export', Block::class);

        return Excel::download(new BlocksExport($request), 'blocks-' . now()->format('dmY_His') . '.xlsx');
    }
}
