<?php

namespace Modules\Admin\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Modules\Admin\Entities\Admin;
use Modules\Admin\Http\Requests\AdminRequest;
use Illuminate\Support\Facades\Hash;
use Maatwebsite\Excel\Facades\Excel;
use Modules\Admin\Exports\AdminsExport;

class AdminController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(Admin::class, 'admin');
    }

    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = Admin::query()
                ->filters($request);

            return Admin::makeDataTable($data);
        }

        return view('admin::index');
    }

    public function trashed(Request $request)
    {
        $this->authorize('restore', Admin::class);

        if ($request->ajax()) {
            $data = Admin::onlyTrashed();

            return Admin::makeDataTable($data, trash: true);
        }

        return view('admin::trashed');
    }

    /**
     * Show the form for creating a new resource.
     * @return Renderable
     */
    public function create()
    {
        return view('admin::create', ['admin' => new Admin()]);
    }

    /**
     * Store a newly created admin in storage
     *
     * @param  \Modules\Admin\Http\Requests\AdminRequest $request
     * @param  \Modules\Admin\Entities\Admin $admin
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(AdminRequest $request, Admin $admin)
    {
        $new_password = $request->get('password');

        $new_admin = $admin->create(array_merge($request->validated(), [
            'picture' => upload_file($request, 'photo', 'avatars'),
            'is_active' => $request->boolean('is_active'),
            'password' => Hash::make($new_password),
        ]));

        return redirect()->route('admin.index')->withStatus(__('New admin successfully created.'));
    }

    /**
     * Show the form for editing the specified admin
     *
     * @param  \Modules\Admin\Entities\Admin $admin
     * @return \Illuminate\View\View
     */
    public function edit(Admin $admin)
    {
        return view('admin::edit', ['admin' => $admin]);
    }

    /**
     * Update the specified admin in storage
     *
     * @param  \Modules\Admin\Http\Requests\AdminRequest $request
     * @param  \Modules\Admin\Entities\Admin $admin
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(AdminRequest $request, Admin $admin)
    {
        $input = array_merge($request->validated(), [
            'picture' => upload_file($request, 'photo', 'avatars', $admin->picture),
            'is_active' => $request->boolean('is_active'),
        ]);

        data_forget($input, 'password');

        $admin->update($input);

        if ($request->filled('password')) {
            $admin->password = Hash::make($request->get('password'));
            $admin->save();
        }

        return redirect()->route('admin.index')->withStatus(__("Admin #$admin->id successfully updated."));
    }

    /**
     * Remove the specified admin from storage
     *
     * @param  \Modules\Admin\Entities\Admin $admin
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Admin $admin)
    {
        $admin->delete();

        return redirect()->route('admin.index')->withStatus(__("Admin #$admin->id successfully deleted."));
    }

    public function restore($id)
    {
        $admin = Admin::onlyTrashed()->findOrFail($id);

        $this->authorize('restore', $admin);

        if ($admin->trashed()) {
            $admin->restore();

            return redirect()->route('admin.trashed')->withStatus(__("Admin #$admin->id successfully restored."));
        }

        return redirect()->route('admin.trashed')->withErrors(__("Admin #$admin->id cannot be restored."));
    }

    public function export(Request $request)
    {
        $this->authorize('export', Admin::class);

        return Excel::download(new AdminsExport($request), 'admins-' . now()->format('dmY_His') . '.xlsx');
    }
}
