<?php

namespace Modules\Admin\Entities;

use App\Notifications\ResetPassword;
use Deligoez\LaravelModelHashId\Traits\HasHashId;
use Deligoez\LaravelModelHashId\Traits\HasHashIdRouting;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Modules\Admin\Database\Factories\AdminFactory;
use App\Traits\QueryCacheable;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Modules\User\Entities\User;
use Nwidart\Modules\Facades\Module;
use Str;
use Yajra\DataTables\Facades\DataTables;

class Admin extends Authenticatable implements MustVerifyEmail
{
    use Notifiable;
    use HasFactory;
    use HasHashId;
    use HasHashIdRouting;
    use SoftDeletes;
    use QueryCacheable;

    /**
     * Specify the amount of time to cache queries.
     * Do not specify or set it to null to disable caching.
     *
     * @var int|\DateTime
     */
    public $cacheFor = 3600;

    /**
     * The tags for the query cache. Can be useful
     * if flushing cache for specific tags only.
     *
     * @var null|array
     */
    public $cacheTags = ['admins'];

    /**
     * Invalidate the cache automatically
     * upon update in the database.
     *
     * @var bool
     */
    protected static $flushCacheOnUpdate = true;

    const SUPERADMIN_ID = 1;
    const PERMISSION_SUPERADMIN = 'superadmin';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'picture',
        'gender',
        'email',
        'contact_no',
        'permissions',
        'password',
        'is_active',
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password',
        'remember_token',
        // 'deleted_at',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'is_active' => 'boolean',
        'permissions' => 'array',
        'updated_at' => 'datetime:Y-m-d H:i:s',
        'created_at' => 'datetime:Y-m-d H:i:s',
        'deleted_at' => 'datetime:Y-m-d H:i:s',
        'email_verified_at' => 'datetime:Y-m-d H:i:s',
        'last_login_at' => 'datetime:Y-m-d H:i:s',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = ['avatar_path'];

    /**
     * Create a new factory instance for the model.
     */
    protected static function newFactory()
    {
        return AdminFactory::new();
    }

    /**
     * Get the path to the profile picture
     *
     * @return string
     */
    public function getAvatarPathAttribute()
    {
        $default = uno()->avatar;

        if ($this->picture) {
            if (filter_var($this->picture, FILTER_VALIDATE_URL)) {
                return $this->picture;
            } else if (asset_exists($this->picture)) {
                return disk_public()->url($this->picture);
            } else {
                return $default;
            }
        }

        return $default;
    }

    /**
     * Check if the user is superadmin
     *
     * @return boolean
     */
    public function isSuperAdmin()
    {
        return $this->id == self::SUPERADMIN_ID || in_array(Admin::PERMISSION_SUPERADMIN, $this->permissions ?: []);
    }

    /**
     * Send the password reset notification.
     *
     * @param  string  $token
     * @return void
     */
    public function sendPasswordResetNotification($token)
    {
        $this->notify(new ResetPassword($token, 'password.reset'));
    }

    public function scopeActive($query)
    {
        $query->where('admins.is_active', true);
    }

    public function scopeFilters($query, Request $request)
    {
        $query->when($request->has('status'), function ($query) use ($request) {
            return $query->where('is_active', $request->status == 'active');
        });

        $query->when($request->has('range'), function ($query) use ($request) {
            $date_range = explode(' to ', $request->range);

            if (count($date_range) == 2) {
                $from = new Carbon($date_range[0]);
                $to = new Carbon($date_range[1]);

                $query->whereBetween('created_at', [$from->startOfDay(), $to->endOfDay()]);
            } else if (count($date_range) == 1) {
                $query->whereDate('created_at', $date_range[0]);
            }

            return $query;
        });
    }

    public static function makeDataTable($source, $trash = false)
    {
        return DataTables::of($source)
            ->addColumn('avatar', function (self $row) {
                return view('admin.datatables.thumbnail', ['path' => $row->avatar_path]);
            })
            ->addColumn('status', function (self $row) {
                return view('admin.datatables.status', ['active' => $row->is_active]);
            })
            ->addColumn('actions', function (self $row) use ($trash) {
                $actions = ['row' => $row];

                if ($trash) {
                    $actions['restore_route'] = 'admin.restore';
                } else {
                    $actions['update_route'] = 'admin.edit';
                    $actions['delete_route'] = 'admin.destroy';
                }

                return view('admin.datatables.actions', $actions);
            })
            // to fixed yajra weird timezone return
            ->editColumn('created_at', fn($row) => $row->created_at)
            ->editColumn('updated_at', fn($row) => $row->updated_at)
            ->editColumn('deleted_at', fn($row) => $row->deleted_at)
            ->make(true);
    }

    public static function getModulesList()
    {
        $exclude = [
            'country',
            'feedback',
        ];

        $include = [
            'feedbacks',
            'setting',
            'page',
        ];

        $modules = collect(Module::allEnabled())
                        ->keys()
                        ->map(fn ($name) => strtolower($name))
                        ->merge($include)
                        ->reject(fn ($name) => in_array($name, $exclude))
                        ->sort();

        return $modules;
    }

    public function hasPermission($permission)
    {
        $permissions = $this->permissions ?: [];

        if ($this->isSuperAdmin()) {
            return true;
        }

        if (Str::of($permission)->endsWith('.*')) {
            return count(preg_grep('/^' . substr($permission, 0, -2) . '\\./i', $permissions)) > 0;
        } else {
            return in_array($permission, $permissions);
        }
    }
}
